/** @file   decorativeobject.h
 * @brief   Declaration of DecorativeObject - class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/03/16 21:05:33 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_DECORATIVE_OBJECT_H
#define H_WWW_DECORATIVE_OBJECT_H

#include "gameobject.h"

namespace WeWantWar {

/** @class  DecorativeObject
 * @brief   Represents the objects that can be used for decorative purpose.
 * @author  Tomi Lamminsaari
 *
 * DecorativeObject is an object that cannot be controlled by a controller.
 * Its purpose is to just stay where it was placed and get destroyed when
 * it's shot. As an example the beer glasses in the bar could be an instances
 * of this class.
 */
class DecorativeObject : public GameObject
{
public:
  
	/** Constructor.
   * @param   aTypeName         The string representation of the decorative
   *                            object type being created.
   */
	DecorativeObject( const std::string& aTypeName );

	/** Destructor
   */
	virtual ~DecorativeObject();

private:
  DecorativeObject( const DecorativeObject& rO );
  DecorativeObject& operator = ( const DecorativeObject& rO );
public:

  /** Updates this object.
   */
  virtual void update();
  
  /** Redraws this object
   * @param     aQueue            pointer to RedrawQueue
   */
  virtual void redraw( RedrawQueue* aQueue );
  
  /** Handles the bullethits
   * @param     aBullet           Pointer to bullet that hit us.
   * @return    <code>true</code> if we accepted the bullet hit.
   */
  virtual bool hitByBullet( Bullet* aBullet );
  
  /** Makes the sound.
   * @param     aSoundId          Id-code of the sound we should make
   */
  virtual void makeSound( GameObject::SoundID aSoundId ) const;

  /** Kills this object.
   */
  virtual void kill();
  
  ///
  /// Getter methods
  /// ==============

  /** Returns the type of this object.
   * @return    ObjectID::TYPE_DECORATIVEOBJECT
   */
  virtual ObjectID::Type objectType() const;
  
  /** Tells if we're reloading
   * @return    Always false
   */
  virtual bool reloading() const;

protected:

  /** Initializes the object based on the type of us.
   */
  void initObject();
  
  /** Sets the correct animation based on the type of this object.
   * @param     aAnimId           Id-code of the animation to set up.
   */
  void setCorrectAnimation( int aAnimId );
  
protected:  // members
  /** A flag that tells if bullets can hit us. */
  bool iAcceptBulletHits;
  
  /** Index of the sound we play when this object gets destroyed. */
  int iDestroySound;
  
  /** Uid of the animation container that contains the animations of this
   * object.
   */
  int iAnimationUid;
  
  /** The string representation of the type of this object. Operates as a prefix
   * to the general settings file.
   */
  std::string iObjectTypeName;
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: decorativeobject.h,v $
 * Revision 1.3  2006/03/16 21:05:33  lamminsa
 * Removed the code that came obsolete when these objects were
 * made more dynamic and flexible.
 *
 * Revision 1.2  2006/02/17 23:08:53  lamminsa
 * no message
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.2  2006-01-10 22:54:27+02  lamminsa
 * EChair2 type added.
 *
 * Revision 1.1  2005-11-23 00:24:26+02  lamminsa
 * More decorative objects added.
 *
 * Revision 1.0  2005-11-06 01:16:54+02  lamminsa
 * Initial revision
 *
 */
 
